/* $Id: tlevel.c,v 1.1 1999/02/02 00:20:40 ericb Exp $ */
/* Copyright (C) 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Template for E1432 test programs */

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"

#define	NMOD_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * E1432_TACH_CHANS)

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: tlevel.c,v 1.1 1999/02/02 00:20:40 ericb Exp $";
static const char *progname;

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw,
     int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].tach_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
    {
	chan_list[i] = E1432_TACH_CHAN(i + 1);
	CHECK(e1432_set_active(hw, chan_list[i], E1432_CHANNEL_ON));
    }

    return 0;
}

static int
set_trig_slope(E1432ID hw, int nchan, SHORTSIZ16 *chan_list,
	       int slope, int verbose)
{
    int     i;

    if (verbose > 0)
	(void) printf("Setting: slope %s\n",
		      slope == E1432_TRIGGER_SLOPE_POS ? "Pos" :
		      slope == E1432_TRIGGER_SLOPE_NEG ? "Neg" :
		      "Unknown");
    for (i = 0; i < nchan; i++)
	CHECK(e1432_set_trigger_slope(hw, chan_list[i], slope));

    return 0;
}

static int
set_trig_llev(E1432ID hw, int nchan, SHORTSIZ16 *chan_list,
	      double llev, int verbose)
{
    int     i;

    if (verbose > 0)
	(void) printf("Setting: lower %f\n", llev);
    for (i = 0; i < nchan; i++)
	CHECK(e1432_set_trigger_level(hw, chan_list[i],
				      E1432_TRIGGER_LEVEL_LOWER,
				      llev));

    return 0;
}

static int
set_trig_ulev(E1432ID hw, int nchan, SHORTSIZ16 *chan_list,
	      double ulev, int verbose)
{
    int     i;

    if (verbose > 0)
	(void) printf("Setting: upper %f\n", ulev);
    for (i = 0; i < nchan; i++)
	CHECK(e1432_set_trigger_level(hw, chan_list[i],
				      E1432_TRIGGER_LEVEL_UPPER,
				      ulev));

    return 0;
}

static int
check_trig_lev(E1432ID hw, int nchan, SHORTSIZ16 *chan_list,
	       char *str, double *llev, double *ulev, int verbose)
{
    FLOATSIZ32 llevel, ulevel;
    int     i;

    if (verbose > 0)
	(void) printf(" %s:\n", str);
    for (i = 0; i < nchan; i++)
    {
	CHECK(e1432_get_trigger_level(hw, chan_list[i],
				      E1432_TRIGGER_LEVEL_LOWER,
				      &llevel));
	CHECK(e1432_get_trigger_level(hw, chan_list[i],
				      E1432_TRIGGER_LEVEL_UPPER,
				      &ulevel));
	if (verbose > 0)
	    (void) printf("  Chan %d: lower %f upper %f\n", i, llevel, ulevel);

	if (fabs(llevel - *llev) > 0.05)
	{
	    (void) fprintf(stderr, "%s: Lower Trigger Level %g, expected %g\n",
			   progname, llevel, *llev);
	    return -1;
	}
	if (fabs(ulevel - *ulev) > 0.05)
	{
	    (void) fprintf(stderr, "%s: Upper Trigger Level %g, expected %g\n",
			   progname, llevel, *llev);
	    return -1;
	}
    }
    if (nchan > 0)
    {
	*llev = llevel;
	*ulev = ulevel;
    }
    else
    {
	*llev = 0;
	*ulev = 0;
    }

    return 0;
}

static int
check_trig_lev2(E1432ID hw, int nchan, SHORTSIZ16 *chan_list,
		char *str, double *llev, double *ulev, int verbose)
{
    double  tmp_llev, tmp_ulev;

    tmp_llev = *llev;
    tmp_ulev = *ulev;

    CHECK(check_trig_lev(hw, nchan, chan_list, str,
			 llev, ulev, verbose));
    CHECK(set_trig_llev(hw, nchan, chan_list, *llev, verbose - 1));
    CHECK(set_trig_ulev(hw, nchan, chan_list, *ulev, verbose - 1));
    CHECK(check_trig_lev(hw, nchan, chan_list, str,
			 &tmp_llev, &tmp_ulev, verbose));

    return 0;
}

static int
run(E1432ID hw, int nchan, SHORTSIZ16 *chan_list, int verbose)
{
    double  llev, ulev;
    int     i;

    llev = 0;
    ulev = 0.05;
    CHECK(check_trig_lev2(hw, nchan, chan_list, "Default",
			  &llev, &ulev, verbose));

    CHECK(set_trig_ulev(hw, nchan, chan_list, 20, verbose));
    ulev = 20;
    llev = 19.75;
    CHECK(check_trig_lev2(hw, nchan, chan_list, "Upper 20V",
			  &llev, &ulev, verbose));

    CHECK(set_trig_llev(hw, nchan, chan_list, 19.5, verbose));
    ulev = 19.75;
    llev = 19.5;
    CHECK(check_trig_lev2(hw, nchan, chan_list, "Lower 19.5V",
			  &llev, &ulev, verbose));

    CHECK(set_trig_ulev(hw, nchan, chan_list, 25, verbose));
    ulev = 25;
    llev = 24.75;
    CHECK(check_trig_lev2(hw, nchan, chan_list, "Upper 25V",
			  &llev, &ulev, verbose));

    /* Catches errors we might not otherwise notice */
    for (i = 0; i < nchan; i++)
	CHECK(e1432_preset(hw, chan_list[i]));

    CHECK(set_trig_ulev(hw, nchan, chan_list, -25, verbose));
    ulev = -24.75;
    llev = -25;
    CHECK(check_trig_lev2(hw, nchan, chan_list, "Upper -25V",
			  &llev, &ulev, verbose));

    /* Catches errors we might not otherwise notice */
    for (i = 0; i < nchan; i++)
	CHECK(e1432_preset(hw, chan_list[i]));

    CHECK(set_trig_ulev(hw, nchan, chan_list, 10, verbose));
    ulev = 10;
    llev = 9.75;
    CHECK(check_trig_lev2(hw, nchan, chan_list, "Upper 10V",
			  &llev, &ulev, verbose));

    CHECK(set_trig_slope(hw, nchan, chan_list,
			 E1432_TRIGGER_SLOPE_NEG, verbose));

    ulev = 10;
    llev = 9.75;
    CHECK(check_trig_lev2(hw, nchan, chan_list, "Slope Neg, Upper 10V",
			  &llev, &ulev, verbose));

    /* Catches errors we might not otherwise notice */
    for (i = 0; i < nchan; i++)
	CHECK(e1432_preset(hw, chan_list[i]));

    CHECK(set_trig_ulev(hw, nchan, chan_list, 20, verbose));
    ulev = 20;
    llev = 19.75;
    CHECK(check_trig_lev2(hw, nchan, chan_list, "Upper 20V",
			  &llev, &ulev, verbose));

    CHECK(set_trig_llev(hw, nchan, chan_list, -4, verbose));
    ulev = -3.95;
    llev = -4;
    CHECK(check_trig_lev2(hw, nchan, chan_list, "Lower -4V",
			  &llev, &ulev, verbose));

    CHECK(set_trig_ulev(hw, nchan, chan_list, 25, verbose));
    ulev = 25;
    llev = 24.75;
    CHECK(check_trig_lev2(hw, nchan, chan_list, "Upper 25V",
			  &llev, &ulev, verbose));

    /* Catches errors we might not otherwise notice */
    for (i = 0; i < nchan; i++)
	CHECK(e1432_preset(hw, chan_list[i]));

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-uvV] [-L laddr] [-n nchan] [-N nmod]\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-n: Use <nchan> channels, default -1 meaning all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    char   *p;
    int     c, i, nchan, nmod, verbose;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    laddr[0] = 8;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "L:n:N:uvV")) != -1)
	switch (c)
	{
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    /* Run the measurement */
    if (init(nmod, laddr, &hw, &nchan, chan_list) < 0)
	return EXIT_FAILURE;
    if (run(hw, nchan, chan_list, verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
